//	TorusGamesUtilities.c
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesUtilities-Common.h"
#include "GeometryGamesLocalization.h"
#include "GeometryGamesHanzi.h"
#include <math.h>
#include <stdio.h>	//	for snprintf() and sscanf()


//	IS_PRECOMPOSED_HANGUL() tests for a precomposed Hangul syllable
//	(unicode range 0xAC00 - 0xD7A3).
#define IS_PRECOMPOSED_HANGUL(c)	((c) >= u'가' && (c) <= u'힣')

//	IS_COMPATIBILITY_JAMO tests for a Hangul compatibility jamo
//	(unicode range 0x3131 - 0x3163).
#define IS_COMPATIBILITY_JAMO(c)	((c) >= u'ㄱ' && (c) <= u'ㅣ')

//	IS_JAMO_CONSONANT() tests for a Hangul compatibility jamo consonant
//	(unicode range range 0x3131 - 0x314E).
#define IS_JAMO_CONSONANT(c)		((c) >= u'ㄱ' && (c) <= u'ㅎ')

//	Unicode's precomposed Hangul syllables are packed systematically
//	into a 3-dimensional array, indexed by the leading consonant (L),
//	vowel (V) and trailing consonant (T).  The trailing consonant may be
//	empty (with index value 0), but the leading consonant and the vowel
//	must be present.

//	The innermost blocks range over T, for fixed L and V.
#define BLOCK_SIZE_T	28
#define BLOCK_SIZE_VT	(21*28)


static Char16	MakeDipthongJamo(Char16 aVowelA, Char16 aVowelB);
static Char16	ComposeHangulSyllable(HangulSyllable *aHangulSyllable);
static Char16	MakeLeadingJamo(Char16 aCompatibilityJamo);
static Char16	MakeVowelJamo(Char16 aCompatibilityJamo);
static Char16	MakeTrailingJamo(Char16 aCompatibilityJamo);

static Char16	*GetGameString(GameType aGame);
static Char16	*GetTopologyString(TopologyType aTopology);

	
void Normalize2DPlacement(
	Placement2D		*aPlacement,
	TopologyType	aTopology)
{
	//	Normalize aPlacement's center to lie within the fundamental domain,
	//	with coordinates in the range -0.5 to +0.5.

	//	Note that the rotation through itsAngle occurs before itsFlip,
	//	so toggling the latter has no effect on the former.

	while (aPlacement->itsV > +0.5)
	{
		aPlacement->itsV -= 1.0;
		if (aTopology == Topology2DKlein)
		{
			aPlacement->itsH	= - aPlacement->itsH;
			aPlacement->itsFlip	= ! aPlacement->itsFlip;
		}
	}
	while (aPlacement->itsV < -0.5)
	{
		aPlacement->itsV += 1.0;
		if (aTopology == Topology2DKlein)
		{
			aPlacement->itsH	= - aPlacement->itsH;
			aPlacement->itsFlip	= ! aPlacement->itsFlip;
		}
	}

	while (aPlacement->itsH > +0.5)
		aPlacement->itsH -= 1.0;
	while (aPlacement->itsH < -0.5)
		aPlacement->itsH += 1.0;
}


void Normalize2DOffset(
	Offset2D		*anOffset,
	TopologyType	aTopology)
{
	//	Normalize anOffset to lie within the fundamental domain,
	//	with coordinates in the range -0.5 to +0.5.

	//	Normalize2DOffset() differs from Normalize2DPlacement()
	//	not only in setting fewer fields, but also in never
	//	negating itsH in a Klein bottle.

	while (anOffset->itsV > +0.5)
	{
		anOffset->itsV -= 1.0;
		if (aTopology == Topology2DKlein)
			anOffset->itsFlip = ! anOffset->itsFlip;
	}
	while (anOffset->itsV < -0.5)
	{
		anOffset->itsV += 1.0;
		if (aTopology == Topology2DKlein)
			anOffset->itsFlip = ! anOffset->itsFlip;
	}

	while (anOffset->itsH > +0.5)
		anOffset->itsH -= 1.0;
	while (anOffset->itsH < -0.5)
		anOffset->itsH += 1.0;
}


double Shortest2DDistance(
	double			h0,
	double			v0,
	double			h1,
	double			v1,
	TopologyType	aTopology,
	bool			*aMirrorFlag)	//	may be NULL
{
	//	Compute the distance between two points in the torus
	//	or Klein bottle.  The coordinates (h0, v0) and (h1, v1)
	//	may lie anywhere in the universal cover.   Conceptually
	//	each point projects down to a point within the fundamental domain,
	//	i.e. with coordinates in the range from -0.5 to +0.5.

	//	Note:  This algorithm finds nearest images iff the objects
	//	are fairly close.  For more distant pairs of objects
	//	the closest images might involve |v1 - v0| > 1/2.

	double	dh,
			dv;

	while (v1 - v0 > +0.5)
	{
		v1 -= 1.0;
		if (aTopology == Topology2DKlein)
		{
			h1 = -h1;
			if (aMirrorFlag != NULL)
				*aMirrorFlag = ! *aMirrorFlag;
		}
	}
	while (v1 - v0 < -0.5)
	{
		v1 += 1.0;
		if (aTopology == Topology2DKlein)
		{
			h1 = -h1;
			if (aMirrorFlag != NULL)
				*aMirrorFlag = ! *aMirrorFlag;
		}
	}

	while (h1 - h0 > +0.5)
		h1 -= 1.0;
	while (h1 - h0 < -0.5)
		h1 += 1.0;

	dh = h1 - h0;
	dv = v1 - v0;

	return sqrt(dh*dh + dv*dv);
}


Char16 BaseLetter(Char16 aLetter)
{
	//	Remove any accents from aLetter and return the "base letter".
	//	For example, 'é', 'è' and 'e' map to 'e'.
	
	//	For comparing kana in Japanese crossword puzzles,
	//	let the hiragana form be the "base letter".
	//	We may assume that small characters have already been converted
	//	to large ones, and that half-width katakana have already been
	//	converted to full-width.
	//	In other words, only full-width large kana should arrive here,
	//	so mapping katakana into hiragana will make the base letter unique.
	
	if (aLetter >= u'ぁ' && aLetter <= u'ん')	//	hiragana
		return aLetter;
	
	if (aLetter >= u'ァ' && aLetter <= u'ン')	//	katakana
		return aLetter - (u'ァ' - u'ぁ');		//		shift to parallel hiragana

	if (aLetter >= u'一' && aLetter <= u'龥')	//	CJK Unified Ideographs (Unicode 4.0)
		return SimplifiedHanzi(aLetter);

	switch (aLetter)
	{
		case u'à':
		case u'á':
		case u'â':
		case u'ã':
		case u'ä':
		case u'ă':
		case u'ạ':
		case u'ả':
		case u'ấ':
		case u'ầ':
		case u'ẩ':
		case u'ẫ':
		case u'ậ':
		case u'ắ':
		case u'ằ':
		case u'ẳ':
		case u'ẵ':
		case u'ặ':
			return u'a';

		case u'è':
		case u'é':
		case u'ê':
		case u'ë':
		case u'ẹ':
		case u'ẻ':
		case u'ẽ':
		case u'ế':
		case u'ề':
		case u'ể':
		case u'ễ':
		case u'ệ':
			return u'e';

		case u'ё':			//	Cyrillic 'yo'
			return u'е';	//	Cyrillic 'ye'

		case u'ì':
		case u'í':
		case u'î':
		case u'ï':
		case u'ĩ':
		case u'ỉ':
		case u'ị':
			return u'i';

		case u'ò':
		case u'ó':
		case u'ô':
		case u'õ':
		case u'ö':
		case u'ő':
		case u'ơ':
		case u'ọ':
		case u'ỏ':
		case u'ố':
		case u'ồ':
		case u'ổ':
		case u'ỗ':
		case u'ộ':
		case u'ớ':
		case u'ờ':
		case u'ở':
		case u'ỡ':
		case u'ợ':
			return u'o';

		case u'ù':
		case u'ú':
		case u'û':
		case u'ü':
		case u'ũ':
		case u'ű':
		case u'ư':
		case u'ụ':
		case u'ủ':
		case u'ứ':
		case u'ừ':
		case u'ử':
		case u'ữ':
		case u'ự':
			return u'u';

		case u'ý':
		case u'ỳ':
		case u'ỵ':
		case u'ỷ':
		case u'ỹ':
			return u'y';

		case u'ç':
			return u'c';

		case u'đ':
			return u'd';

		case u'ñ':
			return u'n';

		default:
			return aLetter;
	}
}


Char16 PromoteHalfWidthKatakana(Char16 aCharacter)
{
	//	The following half-width characters occupy the Unicode range
	//	from 0xFF61 though 0xFF9F
	//
	//		｡｢｣､･ｦｧｨｩｪｫｬｭｮｯｰｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐﾑﾒﾓﾔﾕﾖﾗﾘﾙﾚﾛﾜﾝﾞﾟ
	//
	//	Map them to their full-width equivalents
	//
	//		。「」、・ヲァィゥェォャュョッーアイウエオカキクケコサシスセソタチツテトナニヌネノハヒフヘホマミムメモヤユヨラリルレロワン゛゜
	//

	static Char16	theFullWidthEquivalents[] = u"。「」、・ヲァィゥェォャュョッーアイウエオカキクケコサシスセソタチツテトナニヌネノハヒフヘホマミムメモヤユヨラリルレロワン゛゜";

	if (aCharacter >= u'｡'
	 && aCharacter <= u'ﾟ')
	{
		aCharacter = theFullWidthEquivalents[aCharacter - u'｡'];
	}
	
	return aCharacter;
}


Char16 PromoteSmallKana(Char16 aCharacter)
{
	switch (aCharacter)
	{
		//	hiragana
		case u'ぁ':	return u'あ';
		case u'ぃ':	return u'い';
		case u'ぅ':	return u'う';
		case u'ぇ':	return u'え';
		case u'ぉ':	return u'お';
		case u'っ':	return u'つ';
		case u'ゃ':	return u'や';
		case u'ゅ':	return u'ゆ';
		case u'ょ':	return u'よ';
		case u'ゎ':	return u'わ';
		
		//	katakana
		case u'ァ':	return u'ア';
		case u'ィ':	return u'イ';
		case u'ゥ':	return u'ウ';
		case u'ェ':	return u'エ';
		case u'ォ':	return u'オ';
		case u'ッ':	return u'ツ';
		case u'ャ':	return u'ヤ';
		case u'ュ':	return u'ユ';
		case u'ョ':	return u'ヨ';
		case u'ヮ':	return u'ワ';
		case u'ヵ':	return u'カ';	//	not sure what small カ is for
		case u'ヶ':	return u'ケ';	//	not sure what small ケ is for

		//	half-width katakana (seldom used?)
		case u'ｧ':	return u'ｱ';  
		case u'ｨ':	return u'ｲ';
		case u'ｩ':	return u'ｳ';
		case u'ｪ':	return u'ｴ';
		case u'ｫ':	return u'ｵ';
		case u'ｯ':	return u'ﾂ';
		case u'ｬ':	return u'ﾔ';
		case u'ｭ':	return u'ﾕ';
		case u'ｮ':	return u'ﾖ';
		
		default:	return aCharacter;
	}
}


Char16 RemoveTonos(Char16 aCharacter)
{
	switch (aCharacter)
	{
		case u'Ά':	return u'Α';
		case u'Έ':	return u'Ε';
		case u'Ή':	return u'Η';
		case u'Ί':	return u'Ι';
		case u'Ό':	return u'Ο';
		case u'Ύ':	return u'Υ';
		case u'Ώ':	return u'Ω';
		case u'ά':	return u'α';
		case u'έ':	return u'ε';
		case u'ή':	return u'η';
		case u'ί':	return u'ι';
		case u'ό':	return u'ο';
		case u'ύ':	return u'υ';
		case u'ώ':	return u'ω';
		case u'ΐ':	return u'ι';	//	also removes dialytika

		default:				return aCharacter;
	}
}

Char16 ConvertFinalSigma(Char16 aCharacter)
{
	if (aCharacter == u'ς')
		return u'σ';
	else
		return aCharacter;
}


Char16 ComposeHiragana(
	Char16		aCharacter,
	Char16		*aPendingCharacter)
{
	Char16	thePreviousPendingCharacter;
	
	//	Read *aPendingCharacter then clear it immediately, 
	//	so we don't have to worry about clearing it
	//	in the various cases of the switch statement.
	thePreviousPendingCharacter = *aPendingCharacter;
	*aPendingCharacter = NO_PENDING_CHARACTER;

	switch (thePreviousPendingCharacter)
	{
		case NO_PENDING_CHARACTER:
			switch (aCharacter)
			{
				case u'a':	return u'あ';
				case u'i':	return u'い';
				case u'u':	return u'う';
				case u'e':	return u'え';
				case u'o':	return u'お';
				
				case u'b':
				case u'c':
				case u'd':
				case u'f':
				case u'g':
				case u'h':
				case u'j':
				case u'k':
				case u'm':
				case u'n':
				case u'p':
				case u'r':
				case u's':
				case u't':
				case u'v':
				case u'w':
				case u'y':
				case u'z':
					*aPendingCharacter = aCharacter;
					return u' ';
				
				default:
					return aCharacter;
			}
		
		case u'b':
			switch (aCharacter)
			{
				case u'a':	return u'ば';
				case u'i':	return u'び';
				case u'u':	return u'ぶ';
				case u'e':	return u'べ';
				case u'o':	return u'ぼ';
				default:	return aCharacter;
			}
		
		case u'c':
			switch (aCharacter)
			{
				case u'h':
					*aPendingCharacter = u'č';	//	facilitates "chi" → 'ち'
					return u' ';

				case u'i':	return u'ち';
				default:	return aCharacter;
			}
		
		case u'č':	//	facilitates "chi" → 'ち'
			switch (aCharacter)
			{
				case u'i':	return u'ち';
				default:	return aCharacter;
			}
		
		case u'd':
			switch (aCharacter)
			{
				case u'a':	return u'だ';
				case u'i':	return u'ぢ';
				case u'u':	return u'づ';
				case u'e':	return u'で';
				case u'o':	return u'ど';
				default:	return aCharacter;
			}
		
		case u'f':
			switch (aCharacter)
			{
				case u'u':	return u'ふ';
				default:	return aCharacter;
			}
		
		case u'g':
			switch (aCharacter)
			{
				case u'a':	return u'が';
				case u'i':	return u'ぎ';
				case u'u':	return u'ぐ';
				case u'e':	return u'げ';
				case u'o':	return u'ご';
				default:	return aCharacter;
			}
		
		case u'h':
			switch (aCharacter)
			{
				case u'a':	return u'は';
				case u'i':	return u'ひ';
				case u'u':	return u'ふ';
				case u'e':	return u'へ';
				case u'o':	return u'ほ';
				default:	return aCharacter;
			}
		
		case u'j':
			switch (aCharacter)
			{
				case u'i':	return u'じ';
				default:	return aCharacter;
			}
		
		case u'k':
			switch (aCharacter)
			{
				case u'a':	return u'か';
				case u'i':	return u'き';
				case u'u':	return u'く';
				case u'e':	return u'け';
				case u'o':	return u'こ';
				default:	return aCharacter;
			}
		
		case u'm':
			switch (aCharacter)
			{
				case u'a':	return u'ま';
				case u'i':	return u'み';
				case u'u':	return u'む';
				case u'e':	return u'め';
				case u'o':	return u'も';
				default:	return aCharacter;
			}
		
		case u'n':
			switch (aCharacter)
			{
				case u'a':	return u'な';
				case u'i':	return u'に';
				case u'u':	return u'ぬ';
				case u'e':	return u'ね';
				case u'o':	return u'の';
				case u'n':	return u'ん';
				case u' ':	return u'ん';
				default:	return aCharacter;
			}
		
		case u'p':
			switch (aCharacter)
			{
				case u'a':	return u'ぱ';
				case u'i':	return u'ぴ';
				case u'u':	return u'ぷ';
				case u'e':	return u'ぺ';
				case u'o':	return u'ぽ';
				default:	return aCharacter;
			}
		
		case u'r':
			switch (aCharacter)
			{
				case u'a':	return u'ら';
				case u'i':	return u'り';
				case u'u':	return u'る';
				case u'e':	return u'れ';
				case u'o':	return u'ろ';
				default:	return aCharacter;
			}
		
		case u's':
			switch (aCharacter)
			{
				case u'a':	return u'さ';
				case u'i':	return u'し';
				case u'u':	return u'す';
				case u'e':	return u'せ';
				case u'o':	return u'そ';

				case u'h':
					*aPendingCharacter = u'š';	//	facilitates "shi" → 'し'
					return u' ';

				default:	return aCharacter;
			}
		
		case u'š':	//	facilitates "shi" → 'し'
			switch (aCharacter)
			{
				case u'i':	return u'し';
				default:	return aCharacter;
			}
		
		case u't':
			switch (aCharacter)
			{
				case u'a':	return u'た';
				case u'i':	return u'ち';
				case u'u':	return u'つ';
				case u'e':	return u'て';
				case u'o':	return u'と';
				case u's':	return u'つ';
				default:	return aCharacter;
			}
		
		case u'w':
			switch (aCharacter)
			{
				case u'a':	return u'わ';
				case u'o':	return u'を';
				default:	return aCharacter;
			}
		
		case u'y':
			switch (aCharacter)
			{
				case u'a':	return u'や';
				case u'u':	return u'ゆ';
				case u'o':	return u'よ';
				default:	return aCharacter;
			}
		
		case u'z':
			switch (aCharacter)
			{
				case u'a':	return u'ざ';
				case u'i':	return u'じ';
				case u'u':	return u'ず';
				case u'e':	return u'ぜ';
				case u'o':	return u'ぞ';
				default:	return aCharacter;
			}
		
		default:
			return aCharacter;
	}
}


Char16 ComposeHangul(
	Char16			aCharacter,
	Char16			*aPendingCharacter,
	HangulSyllable	*aPendingHangulSyllable)
{
	Char16	theComposedCharacter,
			theReinterpretedJamo;

	if (IS_PRECOMPOSED_HANGUL(aCharacter))	//	Is aCharacter a complete, precomposed Hangul syllable?
	{
		//	The Windows Korean IME passes fully formed Hangul syllables,
		//	unlike the macOS Korean Input Method which passes jamo.
		*aPendingHangulSyllable	= EMPTY_HANGUL_SYLLABLE;
		theComposedCharacter	= aCharacter;
	}
	else
	if (aCharacter == ' ' || aCharacter == '\t' || aCharacter == '\r' || aCharacter == '\n')
	{
		//	Treat a space, tab, return or newline
		//	as a request to complete the pending syllable.
		theComposedCharacter	= ComposeHangulSyllable(aPendingHangulSyllable);
		*aPendingHangulSyllable	= EMPTY_HANGUL_SYLLABLE;

		//	The Win32 Korean Input Method Editor (IME) passes 
		//	precomposed Hangul syllables, which is fine, but then it also 
		//	passes along the character (typically a return, newline, tab or space)
		//	that the user typed to terminate the word's last syllable.
		//	To avoid overwriting the next crossword cell with an unwanted space,
		//	return 0 in cases where a return or newline arrived when no jamo were pending.
		if ((aCharacter == '\r' || aCharacter == '\n') && theComposedCharacter == ' ')
			theComposedCharacter = 0;
	}
	else
	if ( ! IS_COMPATIBILITY_JAMO(aCharacter))
	{
		//	Aside from precomposed Hangul syllables and white space, 
		//	the only legal input is unicode's "Hangul compatibility jamo".
		//	Flag illegal input with a '*'.
		*aPendingHangulSyllable	= EMPTY_HANGUL_SYLLABLE;
		theComposedCharacter	= '*';
	}
	else	//	Break into cases according to whether we're expecting
			//	a leading consonant, a vowel or a trailing consonant.
	if (aPendingHangulSyllable->itsLeadingConsonant == 0)
	{
		//	aPendingHangulSyllable is empty.
		
		if (IS_JAMO_CONSONANT(aCharacter))
		{
			aPendingHangulSyllable->itsLeadingConsonant = aCharacter;
			theComposedCharacter = ' ';	//	syllable remains incomplete
		}
		else	//	aCharacter is a vowel jamo
		{
			aPendingHangulSyllable->itsLeadingConsonant	= u'ㅇ';
			aPendingHangulSyllable->itsVowel			= aCharacter;
			theComposedCharacter = ' ';	//	syllable remains (tentatively) incomplete
		}
	}
	else
	if (aPendingHangulSyllable->itsVowel == 0)
	{
		//	aPendingHangulSyllable already contains a leading consonant,
		//	but no vowel.
		
		if (IS_JAMO_CONSONANT(aCharacter))
		{
			//	Something is wrong, because a Hangul syllable should
			//	have a simple (non-compound) leading consonant.
			//	Overwrite the old leading consonant and hope for the best.
			aPendingHangulSyllable->itsLeadingConsonant = aCharacter;
			theComposedCharacter = ' ';	//	syllable remains incomplete
		}
		else	//	aCharacter is a vowel jamo
		{
			aPendingHangulSyllable->itsVowel = aCharacter;
			theComposedCharacter = ' ';	//	syllable remains (tentatively) incomplete
		}
	}
	else
	if (aPendingHangulSyllable->itsTrailingConsonant == 0)
	{
		//	aPendingHangulSyllable already contains a leading consonant
		//	and a vowel, but no trailing consonant.
		
		if (IS_JAMO_CONSONANT(aCharacter))
		{
			aPendingHangulSyllable->itsTrailingConsonant = aCharacter;

			//	The syllable remains nominally incomplete,
			//	because we're not sure whether the trailing consonant
			//	really belongs here, or whether it's intended
			//	as the leading consonant for the next syllable.
			theComposedCharacter = ' ';
		}
		else	//	aCharacter is a vowel jamo
		{
			aPendingHangulSyllable->itsVowel = MakeDipthongJamo(aPendingHangulSyllable->itsVowel, aCharacter);
			theComposedCharacter = ' ';	//	syllable remains (tentatively) incomplete
		}
	}
	else
	{
		//	aPendingHangulSyllable already contains a leading consonant,
		//	a vowel and a trailing consonant.
		
		if (IS_JAMO_CONSONANT(aCharacter))
		{
			//	Here we go against the usual Hangul input conventions.
			//	Conventionally, two trailing jamo, for example ㄹ and ㄱ,
			//	get merged to a single trailing jamo ㄺ.  However, such combinations
			//	seem to be rare, and in particular do not occur in the Korean
			//	Torus Games crossword puzzles.  So rather than forcing the user
			//	to type a vowel to rescue the ㄱ, it seems friendlier to simply
			//	let the ㄱ begin a new syllable immedately.

			theComposedCharacter = ComposeHangulSyllable(aPendingHangulSyllable);

			aPendingHangulSyllable->itsLeadingConsonant		= aCharacter;
			aPendingHangulSyllable->itsVowel				= 0x0000;
			aPendingHangulSyllable->itsTrailingConsonant	= 0x0000;
		}
		else	//	aCharacter is a vowel jamo
		{
			//	Re-interpret the trailing jamo of aPendingHangulSyllable
			//	as the leading jamo in a new syllable.

			theReinterpretedJamo = aPendingHangulSyllable->itsTrailingConsonant;

			aPendingHangulSyllable->itsTrailingConsonant = 0x0000;

			theComposedCharacter = ComposeHangulSyllable(aPendingHangulSyllable);

			aPendingHangulSyllable->itsLeadingConsonant		= theReinterpretedJamo;
			aPendingHangulSyllable->itsVowel				= aCharacter;
			aPendingHangulSyllable->itsTrailingConsonant	= 0x0000;
		}
	}

	*aPendingCharacter = ComposeHangulSyllable(aPendingHangulSyllable);

	return theComposedCharacter;
}

static Char16 MakeDipthongJamo(	//	output in unicode's Hangul Compatibility Jamo range 0x3131 - 0x3163
	Char16	aVowelA,			//	 input in unicode's Hangul Compatibility Jamo range 0x3131 - 0x3163
	Char16	aVowelB)
{
	switch (aVowelA)
	{
		case u'ㅗ':
			switch (aVowelB)
			{
				case u'ㅏ': return u'ㅘ';
				case u'ㅐ': return u'ㅙ';
				case u'ㅣ': return u'ㅚ';
			}
			break;

		case u'ㅜ':
			switch (aVowelB)
			{
				case u'ㅓ': return u'ㅝ';
				case u'ㅔ': return u'ㅞ';
				case u'ㅣ': return u'ㅟ';
			}
			break;

		case u'ㅡ':
			switch (aVowelB)
			{
				case u'ㅣ': return u'ㅢ';
			}
			break;
	}
	
	//	If the program gets to this point,
	//	it means aVowelA and aVowelB don't normally combine.
	//	Return aVowelB and hope for the best.
	return aVowelB;
}

static Char16 ComposeHangulSyllable(	//	output typically in unicode's precomposed Hangul range 0xAC00 - 0xD7A3
	HangulSyllable	*aHangulSyllable)
{
	Char16			theLeadingJamo,		//	in unicode's Hangul Jamo  leading subrange 0x1100 - 0x1112
					theVowelJamo,		//	in unicode's Hangul Jamo    vowel subrange 0x1161 - 0x1175
					theTrailingJamo;	//	in unicode's Hangul Jamo trailing subrange 0x11A8 - 0x11C2
	unsigned int	theLeadingIndex,
					theVowelIndex,
					theTrailingIndex;

	if (aHangulSyllable->itsLeadingConsonant == 0)
	{
		//	aHangulSyllable is empty.
		return ' ';	//	in unicode's ASCII range
	}
	else
	if (aHangulSyllable->itsVowel == 0)
	{
		//	aHangulSyllable contains a leading consonant, but no vowel.
		return aHangulSyllable->itsLeadingConsonant;	//	in unicode's Hangul Compatibility Jamo range
	}
	else
	{
		//	aHangulSyllable already contains a leading consonant and a vowel,
		//	but may or may not contain a trailing consonant.

		theLeadingJamo	= MakeLeadingJamo(aHangulSyllable->itsLeadingConsonant);
		theVowelJamo	= MakeVowelJamo(aHangulSyllable->itsVowel);
		theTrailingJamo	= MakeTrailingJamo(aHangulSyllable->itsTrailingConsonant);	//	handles 0 (to trailing jamo) correctly

		theLeadingIndex		= theLeadingJamo  - 0x1100;
		theVowelIndex		= theVowelJamo    - 0x1161;
		theTrailingIndex	= theTrailingJamo - 0x11A7;
		
		return 0xAC00 + theLeadingIndex*BLOCK_SIZE_VT + theVowelIndex*BLOCK_SIZE_T + theTrailingIndex;
	}
}

static Char16 MakeLeadingJamo(	//	output in unicode's Hangul Jamo leading subrange 0x1100 - 0x1112
	Char16	aCompatibilityJamo)	//	 input in unicode's Hangul Compatibility Jamo range 0x3131 - 0x3163
{
	switch (aCompatibilityJamo)
	{
		case u'ㄱ':	return 0x1100;	//	return a leading ㄱ
		case u'ㄲ':	return 0x1101;	//	return a leading ㄲ
		case u'ㄴ':	return 0x1102;	//	return a leading ㄴ
		case u'ㄷ':	return 0x1103;	//	return a leading ㄷ
		case u'ㄸ':	return 0x1104;	//	return a leading ㄸ
		case u'ㄹ':	return 0x1105;	//	return a leading ㄹ
		case u'ㅁ':	return 0x1106;	//	return a leading ㅁ
		case u'ㅂ':	return 0x1107;	//	return a leading ㅂ
		case u'ㅃ':	return 0x1108;	//	return a leading ㅃ
		case u'ㅅ':	return 0x1109;	//	return a leading ㅅ
		case u'ㅆ':	return 0x110A;	//	return a leading ㅆ
		case u'ㅇ':	return 0x110B;	//	return a leading ㅇ
		case u'ㅈ':	return 0x110C;	//	return a leading ㅈ
		case u'ㅉ':	return 0x110D;	//	return a leading ㅉ
		case u'ㅊ':	return 0x110E;	//	return a leading ㅊ
		case u'ㅋ':	return 0x110F;	//	return a leading ㅋ
		case u'ㅌ':	return 0x1110;	//	return a leading ㅌ
		case u'ㅍ':	return 0x1111;	//	return a leading ㅍ
		case u'ㅎ':	return 0x1112;	//	return a leading ㅎ

		default:	return 0x1100;	//	Input is not valid leading consonant --
									//	  return a leading ㄱ and hope for the best.
	}
}

static Char16 MakeVowelJamo(	//	output in unicode's Hangul Jamo vowel subrange 0x1161 - 0x1175
	Char16	aCompatibilityJamo)	//	 input in unicode's Hangul Compatibility Jamo range 0x3131 - 0x3163
{
	//	If aCompatibilityJamo is a valid vowel...
	if (aCompatibilityJamo >= u'ㅏ'
	 && aCompatibilityJamo <= u'ㅣ')
	{
		//	...shift it to the Hangul Jamo range.
		return 0x1161 + (aCompatibilityJamo -  u'ㅏ');
	}
	else	//	aCompatibilityJamo is not a valid vowel, so...
	{
		//	...return an ㅏ and hope for the best.
		return 0x1161;
	}
}

static Char16 MakeTrailingJamo(	//	output in unicode's Hangul Jamo trailing subrange 0x11A8 - 0x11C2
	Char16	aCompatibilityJamo)	//	 input in unicode's Hangul Compatibility Jamo range 0x3131 - 0x3163
{
	switch (aCompatibilityJamo)
	{
		case 0x0000:	return 0x11A7;	//	No trailing consonant is present.
										//	The unicode precomposed Hangul code tables
										//	will handle this case correctly.

		case u'ㄱ':	return 0x11A8;	//	return a trailing ㄱ
		case u'ㄲ':	return 0x11A9;	//	return a trailing ㄲ
		case u'ㄳ':	return 0x11AA;	//	return a trailing ㄳ
		case u'ㄴ':	return 0x11AB;	//	return a trailing ㄴ
		case u'ㄵ':	return 0x11AC;	//	return a trailing ㄵ
		case u'ㄶ':	return 0x11AD;	//	return a trailing ㄶ
		case u'ㄷ':	return 0x11AE;	//	return a trailing ㄷ
		case u'ㄹ':	return 0x11AF;	//	return a trailing ㄹ
		case u'ㄺ':	return 0x11B0;	//	return a trailing ㄺ
		case u'ㄻ':	return 0x11B1;	//	return a trailing ㄻ
		case u'ㄼ':	return 0x11B2;	//	return a trailing ㄼ
		case u'ㄽ':	return 0x11B3;	//	return a trailing ㄽ
		case u'ㄾ':	return 0x11B4;	//	return a trailing ㄾ
		case u'ㄿ':	return 0x11B5;	//	return a trailing ㄿ
		case u'ㅀ':	return 0x11B6;	//	return a trailing ㅀ
		case u'ㅁ':	return 0x11B7;	//	return a trailing ㅁ
		case u'ㅂ':	return 0x11B8;	//	return a trailing ㅂ
		case u'ㅄ':	return 0x11B9;	//	return a trailing ㅄ
		case u'ㅅ':	return 0x11BA;	//	return a trailing ㅅ
		case u'ㅆ':	return 0x11BB;	//	return a trailing ㅆ
		case u'ㅇ':	return 0x11BC;	//	return a trailing ㅇ
		case u'ㅈ':	return 0x11BD;	//	return a trailing ㅈ
		case u'ㅊ':	return 0x11BE;	//	return a trailing ㅊ
		case u'ㅋ':	return 0x11BF;	//	return a trailing ㅋ
		case u'ㅌ':	return 0x11C0;	//	return a trailing ㅌ
		case u'ㅍ':	return 0x11C1;	//	return a trailing ㅍ
		case u'ㅎ':	return 0x11C2;	//	return a trailing ㅎ

		default:	return 0x1100;	//	Input is not valid trailing consonant --
									//	  return a trailing ㄱ and hope for the best.
	}
}

Char16 QWERTYtoDubeolshik(
	Char16 aCharacter)
{
	//	Interpret a keystroke on a QWERTY keyboard layout as if 
	//	the same key had been pressed on a Dubeolshik (두벌식) layout.
	switch (aCharacter)
	{
		case 'q':	return u'ㅂ';
		case 'w':	return u'ㅈ';
		case 'e':	return u'ㄷ';
		case 'r':	return u'ㄱ';
		case 't':	return u'ㅅ';
		case 'y':	return u'ㅛ';
		case 'u':	return u'ㅕ';
		case 'i':	return u'ㅑ';
		case 'o':	return u'ㅐ';
		case 'p':	return u'ㅔ';

		case 'a':	return u'ㅁ';
		case 's':	return u'ㄴ';
		case 'd':	return u'ㅇ';
		case 'f':	return u'ㄹ';
		case 'g':	return u'ㅎ';
		case 'h':	return u'ㅗ';
		case 'j':	return u'ㅓ';
		case 'k':	return u'ㅏ';
		case 'l':	return u'ㅣ';

		case 'z':	return u'ㅋ';
		case 'x':	return u'ㅌ';
		case 'c':	return u'ㅊ';
		case 'v':	return u'ㅍ';
		case 'b':	return u'ㅠ';
		case 'n':	return u'ㅜ';
		case 'm':	return u'ㅡ';

		case 'Q':	return u'ㅃ';
		case 'W':	return u'ㅉ';
		case 'E':	return u'ㄸ';
		case 'R':	return u'ㄲ';
		case 'T':	return u'ㅆ';
		case 'O':	return u'ㅒ';
		case 'P':	return u'ㅖ';

		default:	return aCharacter;
	}
}


Char16 ComposeCyrillic(
	Char16	anExistingCharacter,	//	already in the puzzle cell
	Char16	aNewCharacter)			//	newly arrived
{
	switch (aNewCharacter)
	{
		case u'a':
			switch (anExistingCharacter)
			{
				case u'ы':	return u'я';
				default:	return u'а';
			}		
		case u'b':	return u'б';
		case u'c':
			switch (anExistingCharacter)
			{
				case u'ш':	return u'щ';
				default:	return u'ч';
			}		
		case u'd':	return u'д';
		case u'e':
			switch (anExistingCharacter)
			{
				case u'е':	return u'э';
				case u'э':	return u'е';
				default:	return u'е';
			}		
		case u'f':	return u'ф';
		case u'g':	return u'г';
		case u'h':
			switch (anExistingCharacter)
			{
				case u'ч':	return u'ч';
				case u'с':	return u'ш';
				case u'щ':	return u'щ';
				case u'з':	return u'ж';
				case u'к':	return u'х';
				default:	return u'?';
			}		
		case u'i':	return u'и';
		case u'j':	return u'й';
		case u'k':	return u'к';
		case u'l':	return u'л';
		case u'm':	return u'м';
		case u'n':	return u'н';
		case u'o':
			switch (anExistingCharacter)
			{
				case u'ы':	return u'ё';
				default:	return u'о';
			}		
		case u'p':	return u'п';
		case u'r':	return u'р';
		case u's':
			switch (anExistingCharacter)
			{
				case u'т':	return u'ц';
				default:	return u'с';
			}		
		case u't':	return u'т';
		case u'u':
			switch (anExistingCharacter)
			{
				case u'ы':	return u'ю';
				default:	return u'у';
			}		
		case u'v':	return u'в';
		case u'w':	return u'ш';
		case u'x':	return u'х';
		case u'y':	return u'ы';
		case u'z':	return u'з';
		case u':':	return u'ё';
		case u'\'':	return u'ь';
		case u'\"':	return u'ъ';

		//	All other characters -- including
		//	directly-typed Cyrillic -- pass through unchanged.
		default:	return aNewCharacter;
	}
}


Char16 ComposeGreek(Char16 aCharacter)
{
	switch (aCharacter)
	{
		case 'a':	return u'α';
		case 'b':	return u'β';
		case 'c':	return u'ψ';
		case 'd':	return u'δ';
		case 'e':	return u'ε';
		case 'f':	return u'φ';
		case 'g':	return u'γ';
		case 'h':	return u'η';
		case 'i':	return u'ι';
		case 'j':	return u'ξ';
		case 'k':	return u'κ';
		case 'l':	return u'λ';
		case 'm':	return u'μ';
		case 'n':	return u'ν';
		case 'o':	return u'ο';
		case 'p':	return u'π';
		case 'r':	return u'ρ';
		case 's':	return u'σ';
		case 't':	return u'τ';
		case 'u':	return u'θ';
		case 'v':	return u'ω';
		case 'w':	return u'σ';	//	return 'σ' not 'ς'
		case 'x':	return u'χ';
		case 'y':	return u'υ';
		case 'z':	return u'ζ';
		
		default:	return aCharacter;
	}
}


Char16 ComposeLatin(
	Char16		aCharacter,
	Char16		*aPendingCharacter)
{
	Char16	thePreviousPendingCharacter;
	
	//	Read *aPendingCharacter then clear it immediately, 
	//	so we don't have to worry about clearing it
	//	in the various cases of the switch statement.
	thePreviousPendingCharacter = *aPendingCharacter;
	*aPendingCharacter = NO_PENDING_CHARACTER;

	switch (thePreviousPendingCharacter)
	{
		case NO_PENDING_CHARACTER:
			switch (aCharacter)
			{
				case u'\'':	*aPendingCharacter = u'´';	return u' ';
				case u'`':	*aPendingCharacter = u'`';	return u' ';
				case u'^':	*aPendingCharacter = u'ˆ';	return u' ';
				case u':':	*aPendingCharacter = u'¨';	return u' ';
				case u'"':	*aPendingCharacter = u'˝';	return u' ';
				case u',':	*aPendingCharacter = u'¸';	return u' ';
				case u'~':	*aPendingCharacter = u'˜';	return u' ';
				
				default:
					return aCharacter;
			}
		
		case u'´':
			switch (aCharacter)
			{
				case u'a':	return u'á';
				case u'e':	return u'é';
				case u'i':	return u'í';
				case u'o':	return u'ó';
				case u'u':	return u'ú';
				default:	return aCharacter;
			}
		
		case u'`':
			switch (aCharacter)
			{
				case u'a':	return u'à';
				case u'e':	return u'è';
				case u'i':	return u'ì';
				case u'o':	return u'ò';
				case u'u':	return u'ù';
				default:	return aCharacter;
			}
		
		case u'ˆ':
			switch (aCharacter)
			{
				case u'a':	return u'â';
				case u'e':	return u'ê';
				case u'i':	return u'î';
				case u'o':	return u'ô';
				case u'u':	return u'û';
				default:	return aCharacter;
			}
		
		case u'¨':
			switch (aCharacter)
			{
				case u'a':	return u'ä';
				case u'e':	return u'ë';
				case u'i':	return u'ï';
				case u'o':	return u'ö';
				case u'u':	return u'ü';
				default:	return aCharacter;
			}
		
		case u'˝':
			switch (aCharacter)
			{
				case u'o':	return u'ő';
				case u'u':	return u'ű';
				default:	return aCharacter;
			}
		
		case u'¸':
			switch (aCharacter)
			{
				case u'c':	return u'ç';
				default:	return aCharacter;
			}
		
		case u'˜':
			switch (aCharacter)
			{
				case u'a':	return u'ã';
				case u'n':	return u'ñ';
				case u'o':	return u'õ';
				default:	return aCharacter;
			}
		
		default:
			return aCharacter;
	}
}


unsigned int GetNumPuzzles(
	GameType		aGame,
	TopologyType	aTopology)
{
	Char16			theKey[64];	//	buffer must accommodate "NumTorusWordSearchPuzzles"
	const Char16	*theNumPuzzlesAsUTF16String;
	char			theNumPuzzlesAsUTF8String[16];	//	don't expect more than 2 digits plus terminating zero,
													//		so even 3 chars would suffice
	unsigned int	theNumPuzzles;

	Strcpy16(theKey, BUFFER_LENGTH(theKey), u"Num"						);
	Strcat16(theKey, BUFFER_LENGTH(theKey), GetTopologyString(aTopology));
	Strcat16(theKey, BUFFER_LENGTH(theKey), GetGameString(aGame)		);
	Strcat16(theKey, BUFFER_LENGTH(theKey), u"Puzzles"					);
	
	theNumPuzzlesAsUTF16String = GetLocalizedText(theKey);
	
	UTF16toUTF8(theNumPuzzlesAsUTF16String, theNumPuzzlesAsUTF8String, BUFFER_LENGTH(theNumPuzzlesAsUTF8String));
	if (sscanf(theNumPuzzlesAsUTF8String, "%u", &theNumPuzzles) != 1)
		theNumPuzzles = 0;
		
	return theNumPuzzles;
}

void GetRawPuzzleData(
	GameType		aGame,
	TopologyType	aTopology,
	unsigned int	aPuzzleIndex,
	Char16			*aBuffer,
	unsigned int	aBufferLength)	//	number of Char16's (not bytes!) available in aBuffer, including terminating zero
{
	ErrorText		theError	= NULL;
	Char16			theKey[32]	= u"";
	char			thePuzzleIndexAsUTF8[8];
	Char16			thePuzzleIndexAsUTF16[8];
	const Char16	*theRawPuzzleData;
	
	if (aBuffer == NULL
	 || aBufferLength == 0)
	{
		theError = u"No output buffer";
		goto CleanUpGetRawPuzzleData;
	}

	snprintf(thePuzzleIndexAsUTF8, BUFFER_LENGTH(thePuzzleIndexAsUTF8), "%d", aPuzzleIndex);
	UTF8toUTF16(thePuzzleIndexAsUTF8, thePuzzleIndexAsUTF16, BUFFER_LENGTH(thePuzzleIndexAsUTF16));

	Strcpy16(theKey, BUFFER_LENGTH(theKey), GetTopologyString(aTopology));
	Strcat16(theKey, BUFFER_LENGTH(theKey), GetGameString(aGame)		);
	Strcat16(theKey, BUFFER_LENGTH(theKey), thePuzzleIndexAsUTF16		);

#ifdef USE_CUSTOM_TORUS_CROSSWORD_FOR_TALK
	if (aGame == Game2DCrossword && aTopology == Topology2DTorus)
		Strcpy16(theKey, BUFFER_LENGTH(theKey), u"TorusCrosswordForTalk");
#endif

#ifdef USE_CUSTOM_KLEIN_BOTTLE_CROSSWORD_FOR_TALK
	if (aGame == Game2DCrossword && aTopology == Topology2DKlein)
		Strcpy16(theKey, BUFFER_LENGTH(theKey), u"KleinCrosswordForTalk");
#endif

	theRawPuzzleData = GetLocalizedText(theKey);
	
	Strcpy16(aBuffer, aBufferLength, theRawPuzzleData);

CleanUpGetRawPuzzleData:

	if (theError != NULL)
	{
		GeometryGamesErrorMessage(theError, u"Internal Error in GetRawPuzzleData()");
		if (aBuffer != NULL
		 && aBufferLength > 0)
		{
			aBuffer[0] = 0;
		}
	}
}

static Char16 *GetGameString(GameType aGame)
{
	switch (aGame)
	{
		case Game2DCrossword:	return u"Crossword";
		case Game2DWordSearch:	return u"WordSearch";
		default:
			GeometryGamesErrorMessage(u"Invalid GameType", u"Internal Error in GetGameString()");
			return u"";
	}
}

static Char16 *GetTopologyString(TopologyType aTopology)
{
	switch (aTopology)
	{
		case Topology2DTorus:	return u"Torus";
		case Topology2DKlein:	return u"Klein";
		default:
			GeometryGamesErrorMessage(u"Invalid TopologyType", u"Internal Error in GetTopologyString()");
			return u"";
	}
}


void RefreshStatusMessageText(ModelData *md)
{
	if (md->itsGameRefreshMessage != NULL)
		(*md->itsGameRefreshMessage)(md);
}

